<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Services\ThemeColorService;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;

class ThemeController extends Controller
{
    protected $themeService;

    public function __construct(ThemeColorService $themeService)
    {
        $this->themeService = $themeService;
    }

    /**
     * Show the theme configuration page
     */
    public function index()
    {
        $currentColors = $this->themeService->getThemeColors();
        $availableSchemes = $this->themeService->getAvailableSchemes();
        
        return view('admin.theme.index', compact('currentColors', 'availableSchemes'));
    }

    /**
     * Update theme colors
     */
    public function updateColors(Request $request)
    {
        $request->validate([
            'primary' => 'required|regex:/^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/',
            'primary_light' => 'nullable|regex:/^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/',
            'primary_dark' => 'nullable|regex:/^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/',
            'background_primary' => 'nullable|regex:/^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/',
            'background_secondary' => 'nullable|regex:/^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/',
            'background_tertiary' => 'nullable|regex:/^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/',
        ], [
            '*.regex' => 'Please enter a valid hex color code (e.g., #FF0000)',
        ]);

        try {
            $success = $this->themeService->updateThemeColors($request->only([
                'primary', 'primary_light', 'primary_dark',
                'background_primary', 'background_secondary', 'background_tertiary'
            ]));

            if ($success) {
                return response()->json([
                    'success' => true,
                    'message' => 'Theme colors updated successfully!',
                    'css' => $this->themeService->generateCssVariables()
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to update theme colors.'
                ], 500);
            }
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        }
    }

    /**
     * Apply a predefined color scheme
     */
    public function applyScheme(Request $request)
    {
        $request->validate([
            'scheme' => 'required|string'
        ]);

        $success = $this->themeService->applyColorScheme($request->scheme);

        if ($success) {
            return response()->json([
                'success' => true,
                'message' => 'Color scheme applied successfully!',
                'css' => $this->themeService->generateCssVariables()
            ]);
        } else {
            return response()->json([
                'success' => false,
                'message' => 'Failed to apply color scheme.'
            ], 400);
        }
    }

    /**
     * Preview theme colors (without saving)
     */
    public function previewColors(Request $request)
    {
        $request->validate([
            'primary' => 'required|regex:/^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/',
            'primary_light' => 'nullable|regex:/^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/',
            'primary_dark' => 'nullable|regex:/^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/',
        ]);

        // Generate CSS for preview without saving
        $tempColors = array_merge(
            $this->themeService->getThemeColors(),
            $request->only(['primary', 'primary_light', 'primary_dark'])
        );

        // Create temporary CSS
        $previewCss = $this->generatePreviewCss($tempColors);

        return response()->json([
            'success' => true,
            'css' => $previewCss
        ]);
    }

    /**
     * Generate CSS for preview
     */
    private function generatePreviewCss(array $colors): string
    {
        $css = ":root {\n";
        
        if (isset($colors['primary'])) {
            $css .= "    --color-primary: {$colors['primary']};\n";
        }
        if (isset($colors['primary_light'])) {
            $css .= "    --color-primary-light: {$colors['primary_light']};\n";
        }
        if (isset($colors['primary_dark'])) {
            $css .= "    --color-primary-dark: {$colors['primary_dark']};\n";
        }

        $css .= "}\n";
        
        return $css;
    }

    /**
     * Reset to default colors
     */
    public function resetToDefault()
    {
        // Apply the default blue scheme
        $success = $this->themeService->applyColorScheme('blue');

        if ($success) {
            return response()->json([
                'success' => true,
                'message' => 'Theme reset to default colors!',
                'css' => $this->themeService->generateCssVariables()
            ]);
        } else {
            return response()->json([
                'success' => false,
                'message' => 'Failed to reset theme colors.'
            ], 500);
        }
    }
}