<?php
namespace App\Livewire\User;

use App\Models\Settings;
use App\Models\Transaction;
use App\Models\User;
use App\Notifications\User\TransferNotification;
use Livewire\Component;

class FundTransfer extends Component
{
    public $show_password = false;
    public $password;
    public $amount;
    public $email;
    public $use_transfer;

    public function mount()
    {
        $use_transfer = Settings::select('use_transfer')->find(1)->use_transfer;
        $this->use_transfer = $use_transfer;
        abort_if(! $use_transfer, 404);
    }

    public function render()
    {
        $template = Settings::select('theme')->find(1)->theme;
        return view("{$template}.fund-transfer")
            ->extends("layouts.{$template}")
            ->title('Fund Transfer');
    }

    public function cancel()
    {
        $this->reset();
    }

    public function transfer()
    {
        abort_if(! $this->use_transfer, 401);

        $this->validate([
            'amount' => ['required', 'numeric', 'min:' . Settings::select('min_transfer')->find(1)->min_transfer],
            'email' => ['required'],
            'password' => ['required', 'current_password'],
        ]);

        $receiver = User::whereAny([
            'email',
            'username',
        ], $this->email)
            ->first();

        $user = User::find(auth()->user()->id);

        if (! $receiver) {
           session()->flash('message', __('t.no_user_found'));
            return;
        }

        if ($receiver->id === $user->id) {
            session()->flash('message', __('t.cannot_send_to_self'));
            return;
        }

        $settings = Settings::find(1);
        $charges = floatval($this->amount) * $settings->transfer_charges / 100;
        $todeduct = floatval($this->amount) + $charges;

        if ($user->account_bal < $todeduct) {
            session()->flash('message', __('t.insufficient_balance', ['currency' => $settings->currency, 'amount' => $todeduct]));
            return;
        }

        // deduct balance from sender
        $user->account_bal -= $todeduct;
        $user->save();

        // credit balance to receiver
        $receiver->account_bal += $this->amount;
        $receiver->save();

        // create history
        Transaction::create([
            'user_id' => $user->id,
            'narration' => "Transfered to {$receiver->name}",
            'amount' => $this->amount,
            'type' => 'Debit',
        ]);

        // create history for receiver
        Transaction::create([
            'user_id' => $receiver->id,
            'narration' => "Transfer received from {$user->name}",
            'amount' => $this->amount,
            'type' => 'Credit',
        ]);

        $message = __('t.fund_received', [
    'currency' => $settings->currency,
    'amount' => $this->amount,
    'name' => $user->name
]);

        
        dispatch(function () use ($receiver, $user, $message) {
            $receiver->notify(new TransferNotification($this->amount, $user->name, $receiver->name, now(), $message));
        })->afterResponse();

        session()->flash('success', __('t.success_transfer', [
    'currency' => $settings->currency,
    'amount' => $this->amount,
    'name' => $receiver->name
]));

        $this->reset();
    }
}